/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.Windows.Forms;
using System.ComponentModel;
using System.Drawing; // For ToolboxBitmap
using System.Drawing.Design;
using System.Reflection;

using Borland.Eco.Subscription;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Handles;
using Borland.Eco.Services;
using Borland.Eco.UmlRt;
using Borland.Eco.AutoContainers;
using Borland.Eco.Windows.Forms;
using Borland.Eco.Globalization;

namespace Borland.Eco.WinForm
{
	/// <summary>
	/// ListAction enumeration lists implemented actions that the EcoListActionExtender offers.
	/// If an ImageList is hooked up to the extender component, the ordinal position of the action in the enumeration
	/// will be used as ImageIndex on the extendee.
	/// </summary>
	public enum ListAction
	{
		///<summary>No action is performed.</summary>
		None,
		///<summary>An element of the type of the list is added.</summary>
		Add,
		///<summary>The current element in the list is deleted.</summary>
		Delete,
		///<summary>The next element in the list is made current.</summary>
		MovePrev,
		///<summary>The previous element in the list is made current.</summary>
		MoveNext,
		///<summary>The first element in the list is made current.</summary>
		MoveFirst,
		///<summary>The last element in the list is made current.</summary>
		MoveLast,
		///<summary>The current element in the list is removed from the list.</summary>
		Unlink,
		///<summary>The current element is reloaded from the persistent store.</summary>
		Reload,
		///<summary>An autoform is showed for the current element.</summary>
		ShowAutoForm,
		///<summary>The current element is moved towards the end of the list. This makes sense if the list is an ordered relation.</summary>
		MoveUp,
		///<summary>The current element is moved towards the beginning of the list. This makes sense if the list is an ordered relation.</summary>
		MoveDown,
		///<summary>The Action specified in ActionExpression is performed.</summary>
		ExecuteAction
		//ToEvent
	}
//	public delegate void ListActionHandler(IObject theObject, IElementCollection elementCollection);

	///<summary>
	///This class extends buttons with new properties; EcoListAction, RootHandle, BindingContext, ActionOCL, EnabledOCL and VisibleOCL.
	///The purpose of these are to provide a code free interface to manipulate lists.
	///</summary>
	[ProvideProperty("EcoListAction", typeof(Component))]
	[ProvideProperty("RootHandle", typeof(Component))]
	[ProvideProperty("BindingContext", typeof(Component))]
	[ProvideProperty("ActionExpression", typeof(Component))]
	[ProvideProperty("EnabledOcl", typeof(Component))]
	[ProvideProperty("VisibleOcl", typeof(Component))]
	[ToolboxBitmap(typeof(EcoListActionExtender), "Borland.Eco.WinForm.EcoListActionExtender.bmp")]
	[ToolboxItem(true)]
	[ToolboxItemFilter("System.Windows.Forms")]
	public class EcoListActionExtender: System.ComponentModel.Component, System.ComponentModel.IExtenderProvider, IStaticContextFromObject
	{
		private class Properties
		{
			private object m_Extendee;

			private ElementHandle m_RootHandle;
			private Control m_BindingContext;
			private string m_ActionExpression = string.Empty;
			private string m_EnabledOcl = string.Empty;
			private string m_VisibleOcl = string.Empty;
			private ListAction m_Action = ListAction.None;

			public Properties(object extendee)
			{
				m_Extendee = extendee;
			}
			public ElementHandle RootHandle
			{
				get { return m_RootHandle; }
				set { m_RootHandle = value; }
			}
			public Control BindingContext
			{
				get { return m_BindingContext; }
				set { m_BindingContext = value; }
			}

			public Control EffectiveBindingContext
			{
				get
				{
					if (BindingContext != null)
						return BindingContext;
					else
						return m_Extendee as Control;
				}
			}
			public string ActionExpression { get { return m_ActionExpression; } set { m_ActionExpression = value; } }
			public string EnabledOcl { get { return m_EnabledOcl; } set { m_EnabledOcl = value; } }
			public string VisibleOcl { get { return m_VisibleOcl; } set { m_VisibleOcl = value; } }
			public ListAction ListAction { get { return m_Action; } set { m_Action = value; } }
			private bool HasEnabledOcl { get { return EnabledOcl != null && EnabledOcl.Length > 0; } }
			private bool HasVisibleOcl { get { return VisibleOcl != null && VisibleOcl.Length > 0; } }
			private EnabledChangedSubscriber m_EnabledChangedSubscriber;
			private VisibleChangedSubscriber m_VisibleChangedSubscriber;
			private CurrencyManagerHandle m_BindingHandle;
			public void SetupBindings()
			{
				if ((!HasEnabledOcl && !HasVisibleOcl) || EffectiveBindingContext == null)
				{
					m_BindingHandle = null;
					if (m_EnabledChangedSubscriber != null)
						m_EnabledChangedSubscriber.Deactivate();
					if (m_VisibleChangedSubscriber != null)
						m_VisibleChangedSubscriber.Deactivate();
				}
				else if (m_BindingHandle == null)
					m_BindingHandle = new CurrencyManagerHandle();
				if (m_BindingHandle != null)
				{
					m_BindingHandle.RootHandle = RootHandle;
					m_BindingHandle.BindingContext = EffectiveBindingContext;

					EnqueueEnabledChanged();
					EnqueueVisibleChanged();
				}
			}
			private static void SetEnabled(object extendee, bool value)
			{
				Control control = extendee as Control;
				if (control != null)
				{
					control.Enabled = value;
					return;
				}
				MenuItem menuItem = extendee as MenuItem;
				if (menuItem != null)
					menuItem.Enabled = value;
			}
			private void EnqueueEnabledChanged()
			{
				System.Windows.Forms.Application.Idle += new EventHandler(HandleEnabledChanged);
			}

			private void EnqueueVisibleChanged()
			{
				System.Windows.Forms.Application.Idle += new EventHandler(HandleVisibleChanged);
			}
			private void HandleEnabledChanged(object sender, EventArgs e)
			{
				System.Windows.Forms.Application.Idle -= new EventHandler(HandleEnabledChanged);

				if (m_EnabledChangedSubscriber != null)
					m_EnabledChangedSubscriber.Deactivate();
				m_EnabledChangedSubscriber = new EnabledChangedSubscriber(this);
				m_BindingHandle.SubscribeToElement(m_EnabledChangedSubscriber);
				IOclService OclService = (IOclService)(m_BindingHandle as IEcoServiceProvider).GetEcoService(typeof(IOclService));
				if ((OclService == null) || !HasEnabledOcl)
					return;
				if (m_BindingHandle.Element == null)
				{
					SetEnabled(m_Extendee, false);
					return;
				}
				IElement value = OclService.EvaluateAndSubscribe(
					m_BindingHandle.Element, EnabledOcl, m_EnabledChangedSubscriber, m_EnabledChangedSubscriber);
				SetEnabled(m_Extendee, true.Equals(value.AsObject));
			}

			private static void SetVisible(object extendee, bool value)
			{
				Control control = extendee as Control;
				if (control != null)
				{
					control.Visible = value;
					return;
				}
				MenuItem menuItem = extendee as MenuItem;
				if (menuItem != null)
					menuItem.Visible = value;
			}
			private void HandleVisibleChanged(object sender, EventArgs e)
			{
				System.Windows.Forms.Application.Idle -= new EventHandler(HandleVisibleChanged);
				if (m_VisibleChangedSubscriber != null)
					m_VisibleChangedSubscriber.Deactivate();
				m_VisibleChangedSubscriber = new VisibleChangedSubscriber(this);
				m_BindingHandle.SubscribeToElement(m_VisibleChangedSubscriber);
				IOclService OclService = (IOclService)(m_BindingHandle as IEcoServiceProvider).GetEcoService(typeof(IOclService));
				if ((OclService == null) || !HasVisibleOcl)
					return;
				if (m_BindingHandle.Element == null)
				{
					SetVisible(m_Extendee, false);
					return;
				}
				IElement value = OclService.EvaluateAndSubscribe(
					m_BindingHandle.Element, VisibleOcl, m_VisibleChangedSubscriber, m_VisibleChangedSubscriber);
				SetVisible(m_Extendee, true.Equals(value.AsObject));
			}

			private class EnabledChangedSubscriber: SubscriberAdapterBase
			{
				protected override void DoReceive(object sender, EventArgs e, object actualSubscriber)
				{
					((Properties)actualSubscriber).EnqueueEnabledChanged();
				}

				public EnabledChangedSubscriber(object subscriber): base(subscriber) {}
			}

			private class VisibleChangedSubscriber: SubscriberAdapterBase
			{
				protected override void DoReceive(object sender, EventArgs e, object actualSubscriber)
				{
					((Properties)actualSubscriber).EnqueueVisibleChanged();
				}

				public VisibleChangedSubscriber(object subscriber): base(subscriber) {}
			}
		}

		private readonly Hashtable m_Properties = new Hashtable();

		private Properties PropertiesForObject(object o)
		{
			Properties result = (Properties)m_Properties[o];
			if (result == null)
			{
				result = new Properties(o);
				m_Properties[o] = result;
			}
			return result;
		}
		///<summary>For internal use.</summary>
		public IStaticContext StaticContextForObject(object o)
		{
			Properties p = (Properties)m_Properties[o];
			if (p != null)
				return p.RootHandle;
			else
				return null;
		}

		#region IExtenderProvider implementation
		///<summary>
		///This method determines which controls are to be considered extensible, i.e. receive new properties.
		///You may override this method in a subclass if you need additional (or other) rules to determine
		///which controls to extend.
		///</summary>
		public virtual bool CanExtend(object extendee)
		{
			return (extendee is Button || extendee is MenuItem);
		}
		#endregion

		#region Provided properties
		#region EcoListAction
		///<summary>
		///Getter for EcoListAction.
		///</summary>
		///<param name="component">component to query for.</param>
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoListAction")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyEcoListAction")]
		[DefaultValue(ListAction.None)]
		public ListAction GetEcoListAction(Component component)
		{
				return PropertiesForObject(component).ListAction;
		}

		private string ActionCaption(ListAction value)
		{
			PropertyInfo pi = this.GetType().GetProperty("Caption" + value.ToString()); // do not localize
			return (string)pi.GetValue(this, null);
		}
		private bool MaySetCaption(Component component)
		{
			string caption = EcoWinFormHelper.CaptionFromComponent(component);
			Control control = component as Control;
			string defaultCaption = control != null ? control.Name : string.Empty;
			return ((caption == defaultCaption) ||
							(caption == CaptionAdd) ||
							(caption == CaptionDelete) ||
							(caption == CaptionMovePrev) ||
							(caption == CaptionMoveNext) ||
							(caption == CaptionMoveFirst) ||
							(caption == CaptionMoveLast) ||
							(caption == CaptionUnlink) ||
							(caption == CaptionReload) ||
							(caption == CaptionShowAutoForm) ||
							(caption == CaptionMoveUp) ||
							(caption == CaptionMoveDown) ||
							(caption == CaptionExecuteAction));
		}
		///<summary>
		///This virtual method hooks up the extended control's event. The default case is to hook the protected method <see cref="OnClick"/>
		///to Click, but you may want some other event hooked, or hook to something else.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="component"/> is null.</exception>
		protected virtual void Hookup(Component component)
		{
			if (component == null) throw new ArgumentNullException("component"); // Do not localize
			Control control = component as Control;
			if (control != null)
				control.Click += new EventHandler(OnClick);
			else
			{
				MenuItem menuItem = component as MenuItem;
				if (menuItem != null)
					menuItem.Click += new EventHandler(OnClick);
			}
		}
		///<summary>
		///This virtual method unhooks up the extended controls event. The default case is to unhook the protected method <see cref="OnClick"/>
		///to Click, but if you have hooked some other event you probably want to override this too, for the sake of symmetry.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		protected virtual void Unhook(Component component)
		{
			if (component == null) throw new ArgumentNullException("component"); // Do not localize
			Control control = component as Control;
			if (control != null)
			{
				control.Click -= new EventHandler(OnClick);
				return;
			}
			MenuItem menuItem = component as MenuItem;
			if (menuItem != null)
				menuItem.Click -= new EventHandler(OnClick);
		}
		/// <summary>
		/// Sets the EcoListAction on the extendee. Also sets caption - if caption has not been changed from a default value
		/// and hooks up/removed ImageList and ImageIndex on extendee.
		/// </summary>
		///<param name="component">Component to extend.</param>
		///<param name="value">The <see cref="ListAction"/> to associate with <paramref name="control"/>.</param>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="component"/> is null.</exception>
		public void SetEcoListAction(Component component, ListAction value)
		{
			if (component == null) throw new ArgumentNullException("component"); // Do not localize
			PropertiesForObject(component).ListAction = value;
			Button button = component as Button;
			if (value == ListAction.None)
			{
				Unhook(component);
				Control control = component as Control;
				EcoWinFormHelper.SetCaptionOnComponent(component, control != null ? control.Name : string.Empty);
				if (button != null && button.ImageList == m_imageList)
				{
					button.ImageList = null;
					button.ImageIndex = 0;
				}
			}
			else
			{
				Hookup(component);
				if (MaySetCaption(component))
					EcoWinFormHelper.SetCaptionOnComponent(component, ActionCaption(value));
				if (button != null)
				{
					if (button.ImageList == null)
						button.ImageList = m_imageList;
					if (button.ImageList == m_imageList)
						button.ImageIndex = (System.Int32)value;
				}
			}
		}
		#endregion
		#region RootHandle
		///<summary>
		///Getter for RootHandle.
		///</summary>
		///<param name="component">Component to query.</param>
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoListAction")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyRootHandle")]
		[DefaultValue(null)]
		[Editor("Borland.Eco.Handles.Design.RootHandleEditor, Borland.Eco.Handles.Design", typeof(UITypeEditor))]
		public ElementHandle GetRootHandle(Component component)
		{
			return PropertiesForObject(component).RootHandle;
		}
		///<summary>
		///Setter for RootHandle
		///</summary>
		///<param name="component">Component to extend.</param>
		///<param name="value">The <see cref="ElementHandle"/> to associate with <paramref name="control"/>.</param>
		public void SetRootHandle(Component component, ElementHandle value)
		{
			PropertiesForObject(component).RootHandle = value;
			UpdateBindings(component);
		}
		#endregion
		#region BindingContext
		///<summary>
		///Returns BindingContext if it is set, else the control itself
		///</summary>
		///<param name="component">Component to query.</param>
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoListAction")]
		[LocalizableDescription(typeof(FormsStringRes), "sBindingContextControl")]
		[DefaultValue(null)]
		public Control GetBindingContext(Component component)
		{
			return PropertiesForObject(component).BindingContext;
		}
		///<summary>
		///Setter for BindingContext.
		///</summary>
		///<param name="component">Component to extend.</param>
		///<param name="value">The Control the <paramref name="control"/> will use as its associated binding context.</param>
		///<exception cref="InvalidOperationException">Thrown if called with <paramref name="value"/> equalling <paramref name="control"/>.</exception>
		public void SetBindingContext(Component component, Control value)
		{
			if (value != component)
				PropertiesForObject(component).BindingContext = value;
			else
				throw new InvalidOperationException(FormsStringRes.sCannotLinkBindingContextToSelf);
			UpdateBindings(component);
		}
		#endregion

		#region ActionExpression
		///<summary>
		///Getter for ActionExpression.
		///ActionExpression is an expression of ECO ActionLanguage that will be executed if the Action-property
		///is set to ExecuteAction
		///</summary>
		///<param name="component">Component to query.</param>
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoListAction")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyActionOcl")]
		[DefaultValue("")]
		[Editor("Borland.Eco.Handles.Design.ActionExpressionEditor, Borland.Eco.Handles.Design", typeof(UITypeEditor))]
		public string GetActionExpression(Component component)
		{
			return PropertiesForObject(component).ActionExpression;
		}
		///<summary>
		///Setter for ActionOcl.
		///</summary>
		///<param name="component">Component to extend.</param>
		///<param name="value">The string representing the ActionExpression with which the <paramref name="control"/> will be associated.</param>
		public void SetActionExpression(Component component, string value)
		{
			PropertiesForObject(component).ActionExpression = value;
		}
		#endregion

		#region EnabledOcl
		///<summary>
		///Getter for EnabledOcl.
		///EnabledOcl is an OCL expression controlling if the extended control should be enabled or not.
		///If left blank, ECO will not modify the control's Enabled property.
		///</summary>
		///<param name="component">Component to query.</param>
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoListAction")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyEnabledOcl")]
		[DefaultValue("")]
		[Editor("Borland.Eco.Handles.Design.OclExpressionEditor, Borland.Eco.Handles.Design", typeof(UITypeEditor))]
		public string GetEnabledOcl(Component component)
		{
			return PropertiesForObject(component).EnabledOcl;
		}
		///<summary>
		///Setter for EnabledOcl.
		///</summary>
		///<param name="component">Component to extend.</param>
		///<param name="value">The OCL expression used that will be bound to the Enabled-property of the <paramref name="control"/>.</param>
		public void SetEnabledOcl(Component component, string value)
		{
			PropertiesForObject(component).EnabledOcl = value;
			UpdateBindings(component);
		}
		#endregion

		#region VisibleOcl
		///<summary>
		///Getter for VisibleOcl.
		///EnabledOcl is an OCL expression controlling if the extended control should be visible or not.
		///If left blank, ECO will not modify the control's Visible property.
		///</summary>
		///<param name="component">Component to query.</param>
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoListAction")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyVisibleOcl")]
		[DefaultValue("")]
		[Editor("Borland.Eco.Handles.Design.OclExpressionEditor, Borland.Eco.Handles.Design", typeof(UITypeEditor))]
		public string GetVisibleOcl(Component component)
		{
			return PropertiesForObject(component).VisibleOcl;
		}
		///<summary>
		///Setter for VisibleOcl.
		///</summary>
		///<param name="component">Component to extend.</param>
		///<param name="value">The OCL expression used that will be bound to the Visible-property of the <paramref name="control"/>.</param>
		public void SetVisibleOcl(Component component, string value)
		{
			PropertiesForObject(component).VisibleOcl = value;
			UpdateBindings(component);
		}
		#endregion

		#endregion

		private void UpdateBindings(Component component)
		{
			if (!DesignMode)
				PropertiesForObject(component).SetupBindings();
		}

		#region Eventhandlers

		private static EcoSpace GetEffectiveEcoSpace(ElementHandle elementHandle)
		{
			EcoSpace Result = EcoWinFormHelper.ConnectedEcoSpace(elementHandle);

			// Continue check legal conditions
			if (Result == null)
				throw new InvalidOperationException(FormsStringRes.sErrorEcoSpaceNotSet);
			if (!Result.Active)
				throw new InvalidOperationException(FormsStringRes.sErrorSystemInactive);

			return Result;
		}

		private void DeleteObject(IObject o)
		{
			if ((o != null) && ((DeleteQuestion == null || DeleteQuestion.Length == 0) ||
					(MessageBox.Show(DeleteQuestion, string.Empty, MessageBoxButtons.OKCancel, MessageBoxIcon.Question) == DialogResult.OK)))
				o.Delete();
		}

		private void Remove(IList list, object o)
		{
			if (o != null)
			{
				if (((UnlinkQuestion == null) || (UnlinkQuestion.Length == 0)) ||
						(MessageBox.Show(UnlinkQuestion, string.Empty, MessageBoxButtons.OKCancel, MessageBoxIcon.Question) == DialogResult.OK))
					list.Remove(o);
			}
		}

		private static void Reload(IObject o)
		{
			if (o == null)
				return;

			IPersistenceService persistenceService = (IPersistenceService)o.ServiceProvider.GetEcoService(typeof(IPersistenceService));

			if (persistenceService != null)
				persistenceService.Unload(o);
		}

		private static void ShowAutoForm(EcoSpace effectiveEcoSpace, IElement element)
		{
			AutoContainerArgs autoContainerArgs = new AutoContainerArgs(effectiveEcoSpace, false, AutoContainerMemberVisibility.ProtectedOrHigher, ContainerReusage.ReuseForClass, false);
			IAutoContainer autoContainer = AutoContainerService.Instance.CreateContainer(element, autoContainerArgs);
			autoContainer.Show(autoContainerArgs);
		}

		private static IElementCollection GetCollectionFromCurrencyManager(CurrencyManager cm)
		{
			IElementCollection collection = null;
			IElementProvider collectionProvider = cm.List as IElementProvider;
			if (collectionProvider != null)
				collection = collectionProvider.Element as IElementCollection;
			if (collection == null)
				throw new InvalidOperationException(FormsStringRes.sHandleHasNoList);
			return collection;
		}

		private static void MoveObject(CurrencyManager cm, IElement element, int delta)
		{
			IElementCollection collection = GetCollectionFromCurrencyManager(cm);
			int insertAt = cm.Position + delta;
			if ((insertAt >= 0) && (insertAt < cm.Count))
			{
				collection.Insert(insertAt, element);
				cm.Position = insertAt;
			}
		}

		private static void PerformAction(IObject o, string actionOcl)
		{
			if (o != null)
			{
				try
				{
					IActionLanguageService actionLanguageService = (IActionLanguageService)o.ServiceProvider.GetEcoService(typeof(IActionLanguageService));
					if (actionLanguageService != null)
						actionLanguageService.Execute(o, actionOcl);
				}
				catch (Exception exception)
				{
					MessageBox.Show(exception.ToString());
				}
			}
		}
		///<summary>
		///Performs the activity specified by the <paramref name="action"/>.
		///</summary>
		///<param name="action">The action to perform.</param>
		///<param name="component">An extended component, required to find the extended properties.</param>
		///<exception cref="InvalidOperationException">Thrown if a the <paramref name="component"/> does not have the RootHandle set.</exception>
		///<exception cref="ArgumentException">Thrown if a CurrencyManager cannot be found linking the <paramref name="control"/> and the RootHandle set to that control.</exception>
		public void ExecuteAction(ListAction action, Component component)
		{
			if (!EffectiveEnabled())
				return;

			Control bindingContext = PropertiesForObject(component).EffectiveBindingContext;
			if (bindingContext == null)
				throw new InvalidOperationException(FormsStringRes.sCannotFindBindingContext);
			object bindingContextKey = PropertiesForObject(component).RootHandle;
			if (bindingContextKey == null)
				throw new InvalidOperationException(FormsStringRes.sHandleNotSet);
			CurrencyManager cm = bindingContext.BindingContext[bindingContextKey] as CurrencyManager;

			if (cm == null)
				throw new ArgumentException(FormsStringRes.sCannotFindBindingContext);

			IElement element = null;
			object objectInCM = null;
			if (cm.Position > -1)
			{
				objectInCM = cm.Current;
				IElementProvider elementProvider = objectInCM as IElementProvider;
				if (elementProvider != null)
					element = elementProvider.Element;
			}

			switch (action)
			{
				case ListAction.None:
					break;
				case ListAction.Add:
					IElementCollection ec = GetCollectionFromCurrencyManager(cm);
					if (ec.SupportsAddNew)
						ec.AddNew();
					break;
				case ListAction.Delete:
					IObject o = element as IObject;
					if (o != null)
						DeleteObject(o);
					break;
				case ListAction.MovePrev:
					cm.Position -= 1;
					break;
				case ListAction.MoveNext:
					cm.Position += 1;
					break;
				case ListAction.MoveFirst:
					cm.Position = 0;
					break;
				case ListAction.MoveLast:
					cm.Position = cm.Count;
					break;
				case ListAction.Unlink:
					Remove(cm.List, objectInCM);
					break;
				case ListAction.Reload:
					IObject obj = element as IObject;
					if (obj != null)
					{
						if (((IStateService)obj.ServiceProvider.GetEcoService(typeof(IStateService))).IsDirty(obj))
							MessageBox.Show(FormsStringRes.sCannotReloadDirtyObject);
						else
							Reload(obj);
					}
					break;
				case ListAction.ShowAutoForm:
					ShowAutoForm(GetEffectiveEcoSpace(PropertiesForObject(component).RootHandle), element);
					break;
				case ListAction.MoveUp:
					MoveObject(cm, element, -1);
					break;
				case ListAction.MoveDown:
					MoveObject(cm, element, 1);
					break;
				case ListAction.ExecuteAction:
					PerformAction(element as IObject, PropertiesForObject(component).ActionExpression);
					break;
//				case ListAction.ToEvent: TriggerEvent(sender, cmh); break;
			}
		}
		protected void OnClick(object sender, EventArgs e)
		{
			Component component = (Component)sender;
			ExecuteAction(GetEcoListAction(component), component);
		}
		#endregion

		#region extender's own properties
		#region Enabled property
		private bool m_Enabled = true;
		///<summary>
		///Controls if the extender component to Enabled. If it is not, none of the connected buttons will perform their action.
		///</summary>
		[Browsable(true)]
		[DefaultValue(true)]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryBehaviour")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyExtenderEnabled")]
		public bool Enabled
		{
			get { return m_Enabled; }
			set { m_Enabled = value; }
		}
		private bool EffectiveEnabled()
		{
			return (Enabled);
		}
		#endregion

		#region ImageList
		ImageList m_imageList;
		///<summary>
		///By hooking up an image list to the extender, each connected button can get an image "automatically" by specifying the ImageIndex property.
		///</summary>
		[Browsable(true)]
		[DefaultValue(null)]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryBehaviour")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyExtenderEnabled")]
		public ImageList ImageList
		{
			get { return m_imageList; }
			set
			{
				ICollection keys = m_Properties.Keys;

				foreach (Component component in keys)
				{
					Button button = component as Button;
					if (button != null && button.ImageList == m_imageList)
					{
						button.ImageList = value;
						button.ImageIndex = (Int32)PropertiesForObject(button).ListAction;
					}
				}
				m_imageList = value;
			}
		}
		#endregion
		#region Captions
		private void UpdateConnectedComponents(ListAction action, string oldText, string newText)
		{
			ICollection keys = m_Properties.Keys;

			foreach (Component c in keys)
			{
				if (PropertiesForObject(c).ListAction == action)
				{
					if (EcoWinFormHelper.CaptionFromComponent(c) == oldText)
						EcoWinFormHelper.SetCaptionOnComponent(c, newText);
				}
			}
		}

		private string m_CaptionAdd = FormsStringRes.sCaptionAdd;
		///<summary>
		///The default caption used when the action is set to Add.
		///</summary>
		[DefaultValue("Add")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionAdd")]
		public string CaptionAdd
		{
			get { return m_CaptionAdd; }
			set { UpdateConnectedComponents(ListAction.Add, m_CaptionAdd, value); m_CaptionAdd = value; }
		}
		private string m_CaptionDelete = FormsStringRes.sCaptionDelete;
		///<summary>
		///The default caption used when the action is set to Delete.
		///</summary>
		[DefaultValue("Delete")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionDelete")]
		public string CaptionDelete
		{
			get { return m_CaptionDelete; }
			set { UpdateConnectedComponents(ListAction.Delete, m_CaptionDelete, value); m_CaptionDelete = value; }
		}
		private string m_CaptionMovePrev = FormsStringRes.sCaptionMovePrev;
		///<summary>
		///The default caption used when the action is set to MovePrev.
		///</summary>
		[DefaultValue("Prev")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionMovePrev")]
		public string CaptionMovePrev
		{
			get { return m_CaptionMovePrev; }
			set { UpdateConnectedComponents(ListAction.MovePrev, m_CaptionMovePrev, value); m_CaptionMovePrev = value; }
		}
		private string m_CaptionMoveNext = FormsStringRes.sCaptionMoveNext;
		///<summary>
		///The default caption used when the action is set to MoveNext.
		///</summary>
		[DefaultValue("Next")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionMoveNext")]
		public string CaptionMoveNext
		{
			get { return m_CaptionMoveNext; }
			set { UpdateConnectedComponents(ListAction.MoveNext, m_CaptionMoveNext, value); m_CaptionMoveNext = value; }
		}
		private string m_CaptionMoveFirst = FormsStringRes.sCaptionMoveFirst;
		///<summary>
		///The default caption used when the action is set to MoveFirst.
		///</summary>
		[DefaultValue("First")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionMoveFirst")]
		public string CaptionMoveFirst
		{
			get { return m_CaptionMoveFirst; }
			set { UpdateConnectedComponents(ListAction.MoveFirst, m_CaptionMoveFirst, value); m_CaptionMoveFirst = value; }
		}
		private string m_CaptionMoveLast = FormsStringRes.sCaptionMoveLast;
		///<summary>
		///The default caption used when the action is set to MoveLast.
		///</summary>
		[DefaultValue("Last")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionMoveLast")]
		public string CaptionMoveLast
		{
			get { return m_CaptionMoveLast; }
			set { UpdateConnectedComponents(ListAction.MoveLast, m_CaptionMoveLast, value); m_CaptionMoveLast = value; }
		}
		private string m_CaptionUnlink = FormsStringRes.sCaptionUnlink;
		///<summary>
		///The default caption used when the action is set to Unlink.
		///</summary>
		[DefaultValue("Unlink")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionUnlink")]
		public string CaptionUnlink
		{
			get { return m_CaptionUnlink; }
			set { UpdateConnectedComponents(ListAction.Unlink, m_CaptionUnlink, value); m_CaptionUnlink = value; }
		}
		private string m_CaptionReload = FormsStringRes.sCaptionReload;
		///<summary>
		///The default caption used when the action is set to Reload.
		///</summary>
		[DefaultValue("Reload")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionReload")]
		public string CaptionReload
		{
			get { return m_CaptionReload; }
			set { UpdateConnectedComponents(ListAction.Reload, m_CaptionReload, value); m_CaptionReload = value; }
		}
		private string m_CaptionShowAutoForm = FormsStringRes.sCaptionShowAutoForm;
		///<summary>
		///The default caption used when the action is set to ShowAutoForm.
		///</summary>
		[DefaultValue("Show Autoform")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionShowAutoForm")]
		public string CaptionShowAutoForm
		{
			get { return m_CaptionShowAutoForm; }
			set { UpdateConnectedComponents(ListAction.ShowAutoForm, m_CaptionShowAutoForm, value); m_CaptionShowAutoForm = value; }
		}
		private string m_CaptionMoveUp = FormsStringRes.sCaptionMoveUp;
		///<summary>
		///The default caption used when the action is set to MoveUp.
		///</summary>
		[DefaultValue("Up")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionMoveUp")]
		public string CaptionMoveUp
		{
			get { return m_CaptionMoveUp; }
			set { UpdateConnectedComponents(ListAction.MoveUp, m_CaptionMoveUp, value); m_CaptionMoveUp = value; }
		}
		private string m_CaptionMoveDown = FormsStringRes.sCaptionMoveDown;
		///<summary>
		///The default caption used when the action is set to MoveDown.
		///</summary>
		[DefaultValue("Down")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyCaptionMoveDown")]
		public string CaptionMoveDown
		{
			get { return m_CaptionMoveDown; }
			set { UpdateConnectedComponents(ListAction.MoveDown, m_CaptionMoveDown, value); m_CaptionMoveDown = value; }
		}
		private string m_CaptionExecuteAction = FormsStringRes.sCaptionAction;
		///<summary>
		///The default caption used when the action is set to Action.
		///</summary>
		[DefaultValue("Perform Action")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyAction")]
		public string CaptionExecuteAction
		{
			get { return m_CaptionExecuteAction; }
			set { UpdateConnectedComponents(ListAction.ExecuteAction, m_CaptionExecuteAction, value); m_CaptionExecuteAction = value; }
		}
		private string m_UnlinkQuestion = FormsStringRes.sUnlinkQuestion;
		///<summary>
		///The default text used when querying user to unlink object.
		///</summary>
		[DefaultValue("Unlink object?")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyUnlinkQuestion")]
		public string UnlinkQuestion
		{
			get { return m_UnlinkQuestion; }
			set { m_UnlinkQuestion = value; }
		}
		private string m_DeleteQuestion = FormsStringRes.sDeleteQuestion;
		///<summary>
		///The default text used when querying user to delete object.
		///</summary>
		[DefaultValue("Delete object?")]
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(FormsStringRes), "sPropertyDeleteQuestion")]
		public string DeleteQuestion
		{
			get { return m_DeleteQuestion; }
			set { m_DeleteQuestion = value; }
		}
		#endregion

		///<summary>
		///Obsolete, for backwards compatibility. No longer does anything.
		///</summary>
		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		[Obsolete("Connect the Extender to a handle using the extendee's RootHandle property. Do not use this property.")]
		public EcoSpace EcoSpace
		{
			set { }
		}
		#endregion
	}
}
